/************************************************************************
 * NAME:	h17.c
 *
 * DESCR:	Support for h17 disk images.
 *
 * NOTES:	There are 2 h17 formats created by David Shaw and
 *		David Wallace.  "h17" is the latest format, and "h17-1"
 *		is the early version of this format.
 ************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include "floppy.h"
#include "error.h"
#include "crc.h"
#include "standard.h"
#include "h17.h"

/* these are the indices for the different pieces of the header	*/
/* as found in the two h17 formats.				*/

#define	H17_HEADER_PROTECT	2
#define	H17_HEADER_VOLUME	3
#define	H17_HEADER_SIDES	4
#define	H17_HEADER_TRACKS	5
#define	H17_HEADER_SECTORS	6

/* the tokens in the h17 format are either sector marks or	*/
/* just a simple description...I can imagine them getting	*/
/* more complicated over time...just make changes here.		*/

struct h17_token_line {
    enum { UNKNOWN,
	   DESCRIPTION,
	   SECTOR }	ttype;
    char		description[256];
    int			side;
    int			track;
    int			sector;
    int			protection;
    int			volume;
};

/************************************************************************
 * NAME:	getline()
 *
 * DESCR:	Get a line from the given file.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
char *
getline(char *buffer, int size, FILE *file)
{
    int	got_return = FALSE;

/*    return(fgets(buffer,size,file));	*/

/* it would be nice if fgets() could be convinced to treat \r as \n	*/
/* until I discover that mechanism, I'll just do it myself as follows	*/
/* (this implementation strips out the \n by-the-way (unlike fgets)	*/

    char	*ptr = buffer;
    int		 c;

    for(;;) {
	c = fgetc(file);

	if ( c == -1) {
	    if (got_return) {
		*ptr = '\0';
		return(buffer);
	    } else {
		return(NULL);
	    }
	}

	if ( c != '\0' ) {	/* all NULLs are simply ignored		*/

	    /* a return is treated as a newline, UNLESS followed by a newline	*/

	    if ( c == '\r' ) {

		got_return = TRUE;

	    } else {

		if (c == '\n') {	/* "got_return" is ignored here		*/
		    *ptr = '\0';
		    return(buffer);
		}

		if (got_return) {	/* saw a return, followed by something	*/
		    ungetc(c,file);	/*    other than newline		*/
		    *ptr = '\0';
		    return(buffer);
		}

		*ptr++ = c;

		if (ptr == buffer + size) {
		    return(buffer);
		}
	    }
	}
    }
}

#define DESCR_TOKEN	";Description: "
#define DATA_TOKEN	";Data: "


/************************************************************************
 * NAME:	data_line_parse()
 *
 * DESCR:	Parse the "Data;" line from the h17 file.  There are 2
 *		basic forms, as well as the possibility that it is in
 *		all caps.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
static int
data_line_parse(char *linebuf,struct h17_token_line *token)
{
    int	count;

    count = sscanf(linebuf,"Side %d, Track %d, Sector %d",
		       &token->side, &token->track, &token->sector);
    if (count == 3) {
	return(1);
    }

    count = sscanf(linebuf,"SIDE %d, TRACK %d, SECTOR %d",
		       &token->side, &token->track, &token->sector);
    if (count == 3) {
	return(1);
    }

    /* there appears to be two different formats for the data line	*/


    count = sscanf(linebuf,"Track %d, Side %d, Sector %d",
		       &token->track, &token->side, &token->sector);
    if (count == 3) {
	return(1);
    }

    count = sscanf(linebuf,"TRACK %d, SIDE %d, SECTOR %d",
		       &token->track, &token->side, &token->sector);
    if (count == 3) {
	return(1);
    }

    return(0);
}

/************************************************************************
 * NAME:	read_token_line()
 *
 * DESCR:	Read in the next line, which should be a token line.
 *		The line type is returned along with the token value
 *		as necessary.
 *
 * ARGS:	
 *
 * RETURNS:	0 if not on a token line, 1 otherwise, -1 on EOF
 *
 * NOTES:	
 ************************************************************************/
static int
read_token_line(FILE *file, struct h17_token_line *token)
{
    char	linebuf[257];
    int		count = 0;
    int		e;

    token->ttype = UNKNOWN;

    for(;;) {
	if (getline(linebuf,sizeof(linebuf),file) == NULL) {
	    return(-1);	/* EOF 	*/
	}

	if (linebuf[0] != '\0') {	/* if not a blank line	*/
	    break;
	}
    }

    if (strncasecmp(linebuf,DESCR_TOKEN,strlen(DESCR_TOKEN)) == 0) {
	token->ttype = DESCRIPTION;
	strcpy(token->description,linebuf+strlen(DESCR_TOKEN));
	return(1);
    } else if (strncasecmp(linebuf,DATA_TOKEN,strlen(DATA_TOKEN)) == 0) {
	token->ttype = SECTOR;
	return(data_line_parse(linebuf+strlen(DATA_TOKEN),token));
    } else {
	return(0);
    }
}
    
/************************************************************************
 * NAME:	read_data_line()
 *
 * DESCR:	Read a line of data in the h17 format.  This gets the
 *		line in, and then scans it into the given token.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- blank lines are skipped
 *		- other problems cause an error return
 ************************************************************************/
static int
read_data_line(FILE *file, int *buffer) 
{
    char	linebuf[101];
    char	e;
    int		count = 0;

    /* get the line in	*/

    for(;;) {
	if (getline(linebuf,sizeof(linebuf),file) == NULL) {
	    return(-1);	/* EOF 	*/
	}

	if (linebuf[0] != '\0') {	/* if not a blank line	*/
	    break;
	}
    }

    /* extract the bytes in octal from that line	*/

    count = sscanf(linebuf,"%o %o %o %o %o %o %o %o %o %o %o %o %o %o %o %o",
		   buffer, buffer+1, buffer+2, buffer+3,
		   buffer+4, buffer+5, buffer+6, buffer+7,
		   buffer+8, buffer+9, buffer+10, buffer+11,
		   buffer+12, buffer+13, buffer+14, buffer+15);

    return((count == 16)?1:0);
}

/************************************************************************
 * NAME:	read_h17_header()
 *
 * DESCR:	Loads the header from the input file.  Scans until it
 *		finds one...which could be whacky if the scan takes us
 *		into the data.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
static int
read_h17_header(FILE *file, struct h17_token_line *token)
{
    int	r;
    int	buf[16];

    if (read_data_line(file,buf) <= 0) {
	return(FALSE);
    }

    token->protection = buf[H17_HEADER_PROTECT];
    token->volume = buf[H17_HEADER_VOLUME];
    token->side = buf[H17_HEADER_SIDES];
    token->track = buf[H17_HEADER_TRACKS];
    token->sector = buf[H17_HEADER_SECTORS];

    return (TRUE);
}

/************************************************************************
 * NAME:	read_h17v1_header()
 *
 * DESCR:	Read in the v1 h17 file header.  This is far less formatted
 *		than the later version.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if what came in COULD be a header, FALSE if it simply
 *		didn't look like useful data or EOF.
 *
 * NOTES:	
 ************************************************************************/
static int
read_h17v1_header(FILE *file, struct h17_token_line *token)
{
    char	linebuf[256*3+1];	/* enough for 256 octal bytes	*/
    char	*ptr;
    int		count;
    int		headerbytes[16];
    int		i;

    /* skip blanks first	*/

    for(;;) {
	if (getline(linebuf,sizeof(linebuf),file) == NULL) {
	    return(-1);	/* EOF 	*/
	}
	
	if (linebuf[0] != '\0' ) {	/* if not a blank line	*/
	    break;
	}
    }

    if (strlen(linebuf) < 16*3) {	/* don't have enough data	*/
	return(FALSE);
    }

    for (i=0, ptr = linebuf; i < 16; i++) {
	headerbytes[i]  = (*ptr++ - '0') * 64;
	headerbytes[i] += (*ptr++ - '0') * 8;
	headerbytes[i] += (*ptr++ - '0');
    }

    token->protection = headerbytes[H17_HEADER_PROTECT];
    token->volume = headerbytes[H17_HEADER_VOLUME];
    token->side = headerbytes[H17_HEADER_SIDES];
    token->track = headerbytes[H17_HEADER_TRACKS];
    token->sector = headerbytes[H17_HEADER_SECTORS];

    return(TRUE);
}

/************************************************************************
 * NAME:	read_h17_sector()
 *
 * DESCR:	Read in a 256 byte sector of data.  Assumes that we're pointing
 *		at the data.  Blank lines, though, are skipped.
 *
 * ARGS:	databuf is assumed to have 256 bytes of int storage
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
static int
read_h17_sector(FILE *file, char *databuf)
{
    int	buf[16];
    int	i, j;
    int	r;

    for (i=0; i < 16; i++) {

	if (read_data_line(file,buf) <= 0 ) {
	    return(FALSE);
	}
	for (j=0; j < 16; j++) {
	    databuf[(i*16)+j] = (unsigned char)buf[j];
	}
    }

    return(TRUE);
}

/************************************************************************
 * NAME:	read_h17v1_sector()
 *
 * DESCR:	Read in a 256 byte sector of data.  Assumes that we're pointing
 *		at the data.
 *
 * ARGS:	databuf is assumed to have 256 bytes of int storage
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
static int
read_h17v1_sector(FILE *file, char *databuf)
{
    char	linebuf[256*3+1];	/* enough for 256 octal bytes	*/
    char	*ptr;
    int		count;
    int		i;
    unsigned int value;

    /* skip blanks first	*/

    for(;;) {
	if (getline(linebuf,sizeof(linebuf),file) == NULL) {
	    return(-1);	/* EOF 	*/
	}
	
	if (linebuf[0] != '\0' ) {	/* if not a blank line	*/
	    break;
	}
    }

    if (strlen(linebuf) < 256*3) {		/* don't have enough data	*/
	return(FALSE);
    }

    for (i=0, ptr = linebuf; i < 256; i++) {
	value  = (*ptr++ - '0') * 64;
	value += (*ptr++ - '0') * 8;
	value += (*ptr++ - '0');
	databuf[i] = (unsigned char)value;
    }

    return(TRUE);
}

int
h17_report(int fd, struct floppy *floppy, int level)
{
    int		 errors;

    printf("H17");

    if (level == 1) {
	printf("\n");
	return;
    }

    if (level == 2 || level == 3 || level == 4) {
	lseek(fd,(off_t)0,SEEK_SET);
	errors = h17_read(fd,floppy);
	floppy_report(floppy,level,errors);
    }

}

int
h17v1_report(int fd, struct floppy *floppy, int level)
{
    int		 errors;

    printf("H17v1");

    if (level == 1) {
	printf("\n");
	return;
    }

    if (level == 2 || level == 3 || level == 4) {
	lseek(fd,(off_t)0,SEEK_SET);
	errors = h17v1_read(fd,floppy);
	floppy_report(floppy,level,errors);
    }

}

/************************************************************************
 * NAME:	h17_guesser()
 *
 * DESCR:	
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if it looks like its in h17 format, FALSE otherwise
 *
 * NOTES:	
 ************************************************************************/
int
h17_guesser(int fd)
{
    struct h17_token_line	token;
    FILE			*file;
    int				 ret = FALSE;

    file = fdopen(dup(fd),"r");

    if (read_h17_header(file, &token)) {
	if (token.side >= 1 && token.side <= 2) {
	    if (token.track >=20 && token.track <=80) {
		if (token.sector == 10) {
		    ret = TRUE;
		}
	    }
	}
    }

    fclose(file);
    return(ret);
}

/************************************************************************
 * NAME:	h17v1_guesser()
 *
 * DESCR:	
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if it looks like its in h17v1 format, FALSE otherwise
 *
 * NOTES:	
 ************************************************************************/
int
h17v1_guesser(int fd)
{
    struct h17_token_line	token;
    FILE			*file;
    int				 ret = FALSE;

    file = fdopen(dup(fd),"r");

    if (read_h17v1_header(file, &token)) {
	if (token.side >= 1 && token.side <= 2) {
	    if (token.track >=20 && token.track <=80) {
		if (token.sector == 10) {
		    ret = TRUE;
		}
	    }
	}
    }
    fclose(file);
    return(ret);
}

/************************************************************************
 * NAME:	h17_read()
 *
 * DESCR:	Reads in the h17-format file.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if it read OK, FALSE otherwise.
 *
 * NOTES:	
 ************************************************************************/
int
h17_read(int fd, struct floppy *floppy)
{
    return(h17_read_common(fd,floppy,FALSE));
}

/************************************************************************
 * NAME:	h17v1_read()
 *
 * DESCR:	Reads in the h17v1-format file.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if it read OK, FALSE otherwise.
 *
 * NOTES:	
 ************************************************************************/
int
h17v1_read(int fd, struct floppy *floppy)
{
    return(h17_read_common(fd,floppy,TRUE));
}

/************************************************************************
 * NAME:	h17_read_common()
 *
 * DESCR:	Reads in the h17-format file, either version.
 *
 * ARGS:	if h17v1 is TRUE then we're dealing with the early version
 *
 * RETURNS:	TRUE if it read OK, FALSE otherwise.
 *
 * NOTES:	
 ************************************************************************/
int
h17_read_common(int fd, struct floppy *floppy, int h17v1)
{
    struct h17_token_line	token;
    int				track;
    int 			track_length = 256;	/* should move this to floppy	*/
    int				sector;
    int				side;
    int				tok;
    FILE			*file;

    file = fdopen(dup(fd),"r");

    if (h17v1) {
	if (!read_h17v1_header(file,&token)) {
	    fclose(file);
	    return(E_FATAL|E_H17_BAD_HEADER);
	}
    } else {
	if (!read_h17_header(file,&token)) {
	    fclose(file);
	    return(E_FATAL|E_H17_BAD_HEADER);
	}
    }

    floppy->write_protect = token.protection;
    floppy->volume = token.volume;
    floppy->sides = token.side;
    floppy->tracks = token.track;
    floppy->sectors = token.sector;
    floppy->encoding = H17_HSFM;

    /* get the memory for the number of tracks and sectors	*/

    floppy->side[0] = (struct track *)malloc(floppy->tracks*sizeof(struct track));
    if (floppy->sides == 2) {
	floppy->side[1] = (struct track *)malloc(floppy->tracks*sizeof(struct track));
    }

    for (side = 0; side < floppy->sides; side++) {
	for (track = 0; track < floppy->tracks; track++) {
	    floppy->side[side][track].sector = 
		(struct sector *)malloc(floppy->sectors*sizeof(struct sector));
	    floppy->side[side][track].sectors = floppy->sectors;
	}
    }

    /* for h17v1, we're now positioned at the start of the data	*/
    /* for h17 (the later version) we have some extra stuff	*/

    if (!h17v1) {
	do {
	    tok = read_token_line(file,&token);
	    if (tok < 0) {
		fclose(file);
		return(E_FATAL|E_H17_BAD_FORMAT);
	    }
	} while (tok == 0);

	if (token.ttype == DESCRIPTION) {

	    /* the descrition is simply skipped, read the next token	*/

	    do {
		tok = read_token_line(file,&token);
		if (tok < 0) {
		    fclose(file);
		    return(E_FATAL|E_H17_BAD_FORMAT);
		}
	    } while (tok == 0);
	}
    } else {
	token.ttype = SECTOR;
	token.side = 0;
	token.track = 0;
	token.sector = 0;
    }

    /* found something weird...give up	*/

    if (token.ttype != SECTOR) {
	fclose(file);
	return(E_FATAL|E_H17_BAD_FORMAT);
    }

    /* loop through the sectors, loading them up	*/

    for(;;) {
	if (token.ttype == SECTOR) {
	    if (token.side >= 0 && token.side < floppy->sides ) {
		if( token.track >=0 && token.track < floppy->tracks) {
		    if (token.sector >=0 && token.sector < floppy->sectors) {
			if (h17v1) {
			    if (!read_h17v1_sector(file,floppy->side[token.side][token.track].sector[token.sector].data)) {
				fclose(file);
				return(E_FATAL|E_H17_BAD_SECTOR);
			    }
			} else {
			    if (!read_h17_sector(file,floppy->side[token.side][token.track].sector[token.sector].data)) {
				fclose(file);
				return(E_FATAL|E_H17_BAD_SECTOR);
			    }
			}

			floppy->side[token.side][token.track].sector[token.sector].side = token.side;
			floppy->side[token.side][token.track].sector[token.sector].id = token.track;
			floppy->side[token.side][token.track].sector[token.sector].sector = token.sector;
			floppy->side[token.side][token.track].sector[token.sector].sizecode = floppy->volume;
			floppy->side[token.side][token.track].sector[token.sector].size = 256;
			floppy->side[token.side][token.track].sector[token.sector].mark = 0xfd;
			floppy->side[token.side][token.track].sector[token.sector].headCRC = 0;
			floppy->side[token.side][token.track].sector[token.sector].dataCRC = 0;
			floppy->side[token.side][token.track].sector[token.sector].encoding = H17_HSFM;

			if (token.track == 0) {
			    floppy->side[token.side][token.track].sector[token.sector].sizecode = 0;
			}
		    }
		}
	    }
	}

	if (!h17v1) {
	    if(read_token_line(file,&token) < 0) {
		break;
	    }
	} else {
	    token.sector++;
	    if (token.sector == floppy->sectors) {
		token.sector = 0;
		token.track++;
		if (token.track == floppy->tracks) {
		    break;
		}
	    }
	}
    }

    floppy_crc_set(floppy);	/* set the CRCs	*/

    fclose(file);
    return(E_NONE);
}

/************************************************************************
 * NAME:	h17_dump_header()
 *
 * DESCR:	Dumps an h17 header based upon the data in the floppy
 *		structure.
 *
 * ARGS:	if "v1" is TRUE, dump in h17v1 style.
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
h17_dump_header(int fd, struct floppy *floppy, int v1)
{
    char	buffer[100];
    char *fv1 = "377300%03o%03o%03o%03o%03o000000000000000000000000000\n";
    char *fv2 = "377 300 %03o %03o %03o %03o %03o 000 000 000 000 000 000 000 000 000 \n";

    sprintf(buffer, (v1)?fv1:fv2, floppy->write_protect,
	                          floppy->volume, 
	                          floppy->sides,
	                          floppy->tracks,
	                          floppy->sectors);
    write(fd,buffer,strlen(buffer));
}


/************************************************************************
 * NAME:	h17_dump_sector()
 *
 * DESCR:	Dumps an h17 sector, given a sector data buffer.
 *
 * ARGS:	if "v1" is TRUE, dump in h17v1 style.
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
h17_dump_sector(int fd, int side, int track, int sector, unsigned char *data, int v1)
{
    char	 buffer[1000];
    char	*ptr = buffer;
    int		 i, j;

    if (!v1) {
	sprintf(buffer,
		"\n;DATA: TRACK %03d, SIDE %d, SECTOR %02d\n",
		track, side, sector);
	write(fd,buffer,strlen(buffer));
    }
	
    for (i=0; i < 16; i++) {
	ptr = buffer;
	for (j=0; j < 16; j++) {
	    ptr += sprintf(ptr,"%03o",*data++);
	    if (!v1) {
		*ptr++ = ' ';
	    }
	}
	if (!v1) {
	    *ptr++ = '\n';
	}
	*ptr = '\0';

	write(fd,buffer,strlen(buffer));
    }

    if (v1) {
	write(fd,"\n",1);
    }
}

int
h17_dump_all(int fd, struct floppy *floppy, int v1)
{
    int	track;
    int sector;

    h17_dump_header(fd,floppy,v1);

    for (track = 0; track < floppy->tracks; track++) {
	for (sector = 0; sector < floppy->sectors; sector++) {
	    h17_dump_sector(fd,0,track,sector,LogicalSector(floppy,0,track,sector).data,v1);
	}
    }
}

int
h17v1_dump(int fd, struct floppy *floppy)
{
    return(h17_dump_all(fd,floppy,TRUE));
}

int
h17_dump(int fd, struct floppy *floppy)
{
    return(h17_dump_all(fd,floppy,FALSE));
}

/************************************************************************
 * NAME:	h8d_guesser()
 *
 * DESCR:	"Guesses" the h8d format.  Note, however, that this format
 *		is basically "raw".  So this doesn't do much.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
h8d_guesser(int fd)
{
    struct stat		statbuf;
    int			tracks, slop;

    fstat(fd,&statbuf);

    tracks = statbuf.st_size / (10 * 256);
    slop = statbuf.st_size % (10 * 256);

    if (tracks == 35 || tracks == 40) {
	if (slop == 0) {
	    return(TRUE);
	}
    }
    return(FALSE);
}

/************************************************************************
 * NAME:	h8d_dump()
 *
 * DESCR:	Dumps out an h8d "raw" format.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
h8d_dump(int fd,struct floppy *floppy )
{
    int	track;
    int sector;

    for (track = 0; track < floppy->tracks; track++) {
	for (sector = 0; sector < floppy->sectors; sector++) {
	    write(fd,LogicalSector(floppy,0,track,sector).data,256);
	}
    }
}

/************************************************************************
 * NAME:	h8d_report()
 *
 * DESCR:	Report the structure of an h8d.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
h8d_report(int fd, struct floppy *floppy, int level)
{
    int	errors;

    printf("H8D");

    if (level == 1) {		/* first level, just indicate the type	*/
	printf("\n");
	return;
    }

    if (level == 2 || level == 3 || level == 4) {
	lseek(fd,(off_t)0,SEEK_SET);
	errors = h8d_read(fd,floppy);
	floppy_report(floppy,level,errors);
    }
}

/************************************************************************
 * NAME:	h8d_read()
 *
 * DESCR:	Read in a raw format.  Pretty simple...just a bunch
 *		of tracks.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int h8d_read(int fd, struct floppy *floppy)
{
    int		track;
    int 	sector_length = 256;	/* should move this to floppy	*/
    int		dirtrack;
    int		sector;
    struct stat	statbuf;

#define VOLUME_LOCATION	0

    /* ASSUMES that we have a usable h8d image...so no checking is done	*/
    /*   Only the standard 40 (or 35) tracks are read.  Any beyond that	*/
    /*   are discarded.							*/

    fstat(fd,&statbuf);

    floppy->encoding = H17_HSFM;
    floppy->write_protect = 0;		/* not write-protected		*/
    floppy->sides = 1;
    floppy->sectors = 10;
    floppy->tracks = statbuf.st_size / 256 / 10;
    floppy->volume = 0;

    floppy->side[0] = (struct track *)malloc(floppy->tracks*sizeof(struct track));

    for (track = 0; track < floppy->tracks; track++) {
	floppy->side[0][track].sector = 
	    (struct sector *)malloc(floppy->sectors*sizeof(struct sector));
	floppy->side[0][track].sectors = floppy->sectors;
    }

    for(track =0; track < floppy->tracks; track++) {
	int	count;

	for (sector = 0; sector < floppy->sectors; sector++) {

	    count = read(fd,floppy->side[0][track].sector[sector].data,(size_t)sector_length);

	    if (track == 0 && sector == 9) {

		/* pull out the volume number from deep inside what should be an HDOS	*/
		/* header, or effectively generate a random one by whatever is there.	*/

		floppy->volume = floppy->side[0][track].sector[sector].data[VOLUME_LOCATION];
	    }


	    if (count < sector_length) {
		break;	/* must be done, didn't get a complete track*/
	    }

	    floppy->side[0][track].sector[sector].sizecode = (track == 0)?0:floppy->volume;
	    floppy->side[0][track].sector[sector].id = track;
	    floppy->side[0][track].sector[sector].side = 0;
	    floppy->side[0][track].sector[sector].sector = sector;
	    floppy->side[0][track].sector[sector].size = 256;
	    floppy->side[0][track].sector[sector].mark = 0xfd;
	    floppy->side[0][track].sector[sector].headCRC = 0;
	    floppy->side[0][track].sector[sector].dataCRC = 0;
	    floppy->side[0][track].sector[sector].encoding = floppy->encoding;
	}

	if (count < sector_length) {
	    break;
	}
    }

    floppy_crc_set(floppy);	/* set the CRCs	*/

    return(0);
}


/************************************************************************
 * NAME:	h17_init()
 *
 * DESCR:	Initializes support for the h17-style format.  This
 *		routine registers the call backs for processing h17
 *		including the "guesser", "reading", "writing", and
 *		the report function.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
h17_init()
{
    if (!floppy_format_register("H17","Heathkit H17",h17_guesser,h17_read,h17_dump,h17_report)) {
	return(FALSE);
    }

    if (!floppy_fileext_register("h17", EXT_VERY_SPECIFIC, "H17")) {
	return(FALSE);
    }

    return(TRUE);
}

/************************************************************************
 * NAME:	h17v1_init()
 *
 * DESCR:	Initializes support for the earlier h17-style format.  This
 *		routine registers the call backs for processing h17-1
 *		including the "guesser", "reading", "writing", and
 *		the report function.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
h17v1_init()
{
    if (!floppy_format_register("H17v1","Heathkit H17 version 1", h17v1_guesser, h17v1_read, h17v1_dump, h17v1_report)) {
	return(FALSE);
    }

    if (!floppy_fileext_register("h17", EXT_VERY_SPECIFIC, "H17v1")) {
	return(FALSE);
    }

    return(TRUE);
}

/************************************************************************
 * NAME:	h8d_init()
 *
 * DESCR:	Initializes support for the "raw" style of heathkit disks.
 *		There is no great way to know that these images are FOR SURE
 *		heathkit images...ie - the guesser is not very good.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
h8d_init()
{
    if (!floppy_format_register("H8D","Heathkit Raw Image", h8d_guesser, h8d_read, h8d_dump, h8d_report)) {
	return(FALSE);
    }

    if (!floppy_fileext_register("h8d", EXT_SOMEWHAT_SPECIFIC, "H8D")) {
	return(FALSE);
    }

    if (!floppy_fileext_register("h17", EXT_SOMEWHAT_SPECIFIC, "H8D")) {
	return(FALSE);
    }

    /* NOTE - that the heathkit h8d is NOT put as part of the "DSK" chain	*/
    /* 	it would FOR SURE crash into the JV1 or even JVC formats		*/

    return(TRUE);
}
